<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    public function index()
    {
        return view('dashboard.profile.index', [
            'user' => auth()->user()
        ]);
    }

    public function security()
    {
        return view('dashboard.profile.security');
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . auth()->id()],
        ]);

        $request->user()->update($validated);

        return back()->with('success', 'Profil berhasil diperbarui!');
    }

    public function updatePassword(Request $request)
    {
        $validated = $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', Password::min(8)],
        ]);

        $request->user()->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('success', 'Password berhasil diubah!');
    }

    public function updateSlug(Request $request)
    {
        $validated = $request->validate([
            'security_password' => ['required', 'current_password'],
            'login_slug' => ['required', 'string', 'alpha_num', 'max:50'], // Alpha num only (no dash/space as requested "satu kata")
        ]);

        $request->user()->update([
            'login_slug' => $validated['login_slug']
        ]);

        return back()->with('success', 'URL Login berhasil diubah! Jangan lupa URL baru Anda: ' . url('/login/' . $validated['login_slug']));
    }

    public function updateSecurityQuestion(Request $request)
    {
        // Require password for any change
        $request->validate([
            'security_password' => ['required', 'current_password'],
        ]);

        // Check if enabling or disabling
        $isEnabled = $request->has('is_security_question_enabled');

        if (!$isEnabled) {
            $request->user()->update([
                'is_security_question_enabled' => false,
            ]);
            return back()->with('success', 'Pertanyaan keamanan dinonaktifkan.');
        }

        $validated = $request->validate([
            'security_question' => ['required', 'string'],
            'security_answer' => ['required', 'string', 'max:255'],
        ]);

        $request->user()->update([
            'is_security_question_enabled' => true,
            'security_question' => $validated['security_question'],
            // Normalize answer for case-insensitive check later
            'security_answer' => strtolower(trim($validated['security_answer'])),
        ]);

        return back()->with('success', 'Pertanyaan keamanan berhasil diaktifkan.');
    }
    public function updateHomepage(Request $request)
    {
        $validated = $request->validate([
            'homepage_mode' => ['required', 'in:default,not_found,redirect'],
            'homepage_redirect_url' => ['nullable', 'url', 'required_if:homepage_mode,redirect'],
        ]);

        $request->user()->update([
            'homepage_mode' => $validated['homepage_mode'],
            'homepage_redirect_url' => $validated['homepage_mode'] === 'redirect' ? $validated['homepage_redirect_url'] : null,
        ]);

        return back()->with('success', 'Pengaturan homepage berhasil diperbarui!');
    }

    public function updateHardReset(Request $request)
    {
        // Require password
        $request->validate([
            'security_password' => ['required', 'current_password'],
        ]);

        $isEnabled = $request->has('hard_reset_enabled');

        if (!$isEnabled) {
            $request->user()->update([
                'hard_reset_enabled' => false,
            ]);
            return back()->with('success', 'Fitur Hard Reset dinonaktifkan.');
        }

        $validated = $request->validate([
            'hard_reset_code' => [
                'required',
                'string',
                'min:10',
                'regex:/[a-zA-Z]/', // contains letter
                'regex:/[0-9]/',    // contains number
                'regex:/[\W_]/',    // contains symbol
            ],
        ]);

        $request->user()->update([
            'hard_reset_enabled' => true,
            'hard_reset_code' => $validated['hard_reset_code'],
        ]);

        return back()->with('success', 'Hard Reset Password berhasil diaktifkan.');
    }
}
